// tessellation shader.cpp
#include "TerrainTessellationShader.h"


TerrainTessellationShader::TerrainTessellationShader(ID3D11Device* device, HWND hwnd) : BowerShader(device, hwnd)
{
	initShader(L"bypass_vs.cso", L"terrainTessellation_hs.cso", L"terrainTessellation_ds.cso", L"terrainTessellation_ps.cso");
}


TerrainTessellationShader::~TerrainTessellationShader()
{
	//Release the buffers
	if (matrixBuffer)
	{
		matrixBuffer->Release();
		matrixBuffer = 0;
	}

	if (normalBuffer)
	{
		normalBuffer->Release();
		normalBuffer = 0;
	}

	if (sobelBuffer)
	{
		sobelBuffer->Release();
		sobelBuffer = 0;
	}

	if (dynamicTessBuffer)
	{
		dynamicTessBuffer->Release();
		dynamicTessBuffer = 0;
	}

	if (lightBuffer)
	{
		lightBuffer->Release();
		lightBuffer = 0;
	}

	//Release the samplers
	if (sampleState)
	{
		sampleState->Release();
		sampleState = 0;
	}

	if (sampleStateShadow)
	{
		sampleStateShadow->Release();
		sampleStateShadow = 0;
	}
	
	if (layout)
	{
		layout->Release();
		layout = 0;
	}

	BaseShader::~BaseShader();
}

void TerrainTessellationShader::initShader(const wchar_t* vsFilename, const wchar_t* psFilename)
{
	loadVertexShader(vsFilename);
	loadPixelShader(psFilename);

	//Setup the matrix buffer
	D3D11_BUFFER_DESC matrixBufferDesc;
	matrixBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	matrixBufferDesc.ByteWidth = sizeof(ShadowMatrixBufferType);
	matrixBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	matrixBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	matrixBufferDesc.MiscFlags = 0;
	matrixBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&matrixBufferDesc, NULL, &matrixBuffer);

	//Setup the normal buffer
	D3D11_BUFFER_DESC normalBufferDesc;
	normalBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	normalBufferDesc.ByteWidth = sizeof(NormalsBufferType);
	normalBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	normalBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	normalBufferDesc.MiscFlags = 0;
	normalBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&normalBufferDesc, NULL, &normalBuffer);

	//Setup the sobel buffer
	D3D11_BUFFER_DESC sobelBufferDesc;
	sobelBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	sobelBufferDesc.ByteWidth = sizeof(SobelBufferType);
	sobelBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	sobelBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	sobelBufferDesc.MiscFlags = 0;
	sobelBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&sobelBufferDesc, NULL, &sobelBuffer);

	//Setup the dynamic tessellation buffer
	D3D11_BUFFER_DESC dynamicTessBufferDesc;
	dynamicTessBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	dynamicTessBufferDesc.ByteWidth = sizeof(DynamicTessType);
	dynamicTessBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	dynamicTessBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	dynamicTessBufferDesc.MiscFlags = 0;
	dynamicTessBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&dynamicTessBufferDesc, NULL, &dynamicTessBuffer);

	//Setup the light buffer
	D3D11_BUFFER_DESC lightBufferDesc;
	lightBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
	lightBufferDesc.ByteWidth = sizeof(LightBufferType);
	lightBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
	lightBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
	lightBufferDesc.MiscFlags = 0;
	lightBufferDesc.StructureByteStride = 0;
	renderer->CreateBuffer(&lightBufferDesc, NULL, &lightBuffer);

	//Setup the base sampler
	D3D11_SAMPLER_DESC samplerDesc;
	samplerDesc.Filter = D3D11_FILTER_ANISOTROPIC;
	samplerDesc.AddressU = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.AddressV = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.AddressW = D3D11_TEXTURE_ADDRESS_WRAP;
	samplerDesc.MipLODBias = 0.0f;
	samplerDesc.MaxAnisotropy = 1;
	samplerDesc.ComparisonFunc = D3D11_COMPARISON_ALWAYS;
	samplerDesc.MinLOD = 0;
	samplerDesc.MaxLOD = D3D11_FLOAT32_MAX;
	renderer->CreateSamplerState(&samplerDesc, &sampleState);

	//Setup the shadow sampler
	D3D11_SAMPLER_DESC shadowSamplerDesc;
	shadowSamplerDesc.Filter = D3D11_FILTER_MIN_MAG_MIP_POINT;
	shadowSamplerDesc.AddressU = D3D11_TEXTURE_ADDRESS_BORDER;
	shadowSamplerDesc.AddressV = D3D11_TEXTURE_ADDRESS_BORDER;
	shadowSamplerDesc.AddressW = D3D11_TEXTURE_ADDRESS_BORDER;
	shadowSamplerDesc.BorderColor[0] = 1.0f;
	shadowSamplerDesc.BorderColor[1] = 1.0f;
	shadowSamplerDesc.BorderColor[2] = 1.0f;
	shadowSamplerDesc.BorderColor[3] = 1.0f;
	renderer->CreateSamplerState(&shadowSamplerDesc, &sampleStateShadow);
}

void TerrainTessellationShader::initShader(const wchar_t* vsFilename, const wchar_t* hsFilename, const wchar_t* dsFilename, const wchar_t* psFilename)
{
	initShader(vsFilename, psFilename);
	loadHullShader(hsFilename);
	loadDomainShader(dsFilename);
}

void TerrainTessellationShader::setShaderParameters(ID3D11DeviceContext* deviceContext, const XMMATRIX &worldMatrix, const XMMATRIX &viewMatrix, const XMMATRIX &projectionMatrix, XMFLOAT3 cameraPos, float minTessFactor, float maxTessFactor, float sobelScaleFactor, ID3D11ShaderResourceView* displacementTexture, ID3D11ShaderResourceView* textureMap, ID3D11ShaderResourceView* grassTexture, ID3D11ShaderResourceView* rockTexture, ID3D11ShaderResourceView* dirtTexture, ID3D11ShaderResourceView* sandTexture, ID3D11ShaderResourceView* sunShadowMap, ID3D11ShaderResourceView* spotShadowMap, ID3D11ShaderResourceView* hillShadowMaps[6], float constant, float linear, float quadratic, bool normalMode)
{
	D3D11_MAPPED_SUBRESOURCE mappedResource;
	
	XMMATRIX tworld = XMMatrixTranspose(worldMatrix);
	XMMATRIX tview = XMMatrixTranspose(viewMatrix);
	XMMATRIX tproj = XMMatrixTranspose(projectionMatrix);

	//Map the matrix buffer
	deviceContext->Map(matrixBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	ShadowMatrixBufferType* shadowMatrixPtr = (ShadowMatrixBufferType*)mappedResource.pData;
	shadowMatrixPtr->world = tworld;
	shadowMatrixPtr->view = tview;
	shadowMatrixPtr->projection = tproj;
	shadowMatrixPtr->sunLightView = XMMatrixTranspose(_SunLight->getViewMatrix());
	shadowMatrixPtr->sunLightProjection = XMMatrixTranspose(_SunLight->getOrthoMatrix());

	//Get the view matrix for the spot light
	XMMATRIX spotViewMatrix;
	if (_SpotLight->getDirection().x == 0 && _SpotLight->getDirection().z == 0)
	{
		spotViewMatrix = GetYAxisViewMatrix(_SpotLight);

		if (_SpotLight->getDirection().y < 0.0f)
		{
			spotViewMatrix = -spotViewMatrix;
		}
	}

	else
	{
		_SpotLight->generateViewMatrix();
		spotViewMatrix = _SpotLight->getViewMatrix();
	}

	//Set the values into the mapped pointer
	shadowMatrixPtr->spotLightView = XMMatrixTranspose(spotViewMatrix);
	shadowMatrixPtr->spotLightProjection = XMMatrixTranspose(_SpotLight->getProjectionMatrix());

	//Get the view matrices for the point light
	XMFLOAT3 lightDirections[6] =
	{
		XMFLOAT3(0.0f, 1.0f, 0.0f),
		XMFLOAT3(0.0f, -1.0f, 0.0f),
		XMFLOAT3(1.0f, 0.0f, 0.0f),
		XMFLOAT3(-1.0f, 0.0f, 0.0f),
		XMFLOAT3(0.0f, 0.0f, 1.0f),
		XMFLOAT3(0.0f, 0.0f, -1.0f)
	};

	for (int i = 0; i < 6; i++)
	{
		_HillLight->setDirection(lightDirections[i].x, lightDirections[i].y, lightDirections[i].z);
		XMMATRIX hillViewMatrix;

		if (_HillLight->getDirection().x == 0 && _HillLight->getDirection().z == 0)
		{
			hillViewMatrix = GetYAxisViewMatrix(_HillLight);

			if (_HillLight->getDirection().y < 0.0f)
			{
				hillViewMatrix = -hillViewMatrix;
			}
		}

		else
		{
			_HillLight->generateViewMatrix();
			hillViewMatrix = _HillLight->getViewMatrix();
		}

		XMMATRIX lightViewMatrix = XMMatrixTranspose(hillViewMatrix);
		XMMATRIX lightProjectionMatrix = XMMatrixTranspose(_HillLight->getProjectionMatrix());

		//Set the values into the mapped pointer
		shadowMatrixPtr->hillLightViews[i] = lightViewMatrix;
		shadowMatrixPtr->hillLightProjections[i] = lightProjectionMatrix;
	}

	deviceContext->Unmap(matrixBuffer, 0);
	deviceContext->DSSetConstantBuffers(0, 1, &matrixBuffer);

	//Map the light buffer
	deviceContext->Map(lightBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	LightBufferType* lightPtr = (LightBufferType*)mappedResource.pData;
	lightPtr->sunAmbient = _SunLight->getAmbientColour();
	lightPtr->sunDiffuse = _SunLight->getDiffuseColour();
	lightPtr->sunDirection = _SunLight->getDirection();
	lightPtr->padding = 0.0f;
	lightPtr->spotDiffuse = _SpotLight->getDiffuseColour();
	lightPtr->spotPosition = _SpotLight->getPosition();
	lightPtr->constantFactor = constant;
	lightPtr->linearFactor = linear;
	lightPtr->quadraticFactor = quadratic;
	lightPtr->paddingTwo = XMFLOAT2(0.0f, 0.0f);
	lightPtr->spotDirection = _SpotLight->getDirection();
	lightPtr->paddingThree = 0.0f;
	lightPtr->hillDiffuse = _HillLight->getDiffuseColour();
	lightPtr->hillPosition = _HillLight->getPosition();
	lightPtr->paddingFour = 0.0f;
	deviceContext->Unmap(lightBuffer, 0);
	deviceContext->PSSetConstantBuffers(0, 1, &lightBuffer);

	//Map the normal buffer
	deviceContext->Map(normalBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	NormalsBufferType* normalPtr = (NormalsBufferType*)mappedResource.pData;
	normalMode == true ? normalPtr->normalMode = 1 : normalPtr->normalMode = 0;
	deviceContext->Unmap(normalBuffer, 0);
	deviceContext->PSSetConstantBuffers(1, 1, &normalBuffer);

	//Map the sobel buffer
	deviceContext->Map(sobelBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	SobelBufferType* sobelPtr = (SobelBufferType*)mappedResource.pData;
	sobelPtr->sobelScaleFactor = sobelScaleFactor;
	sobelPtr->padding = XMFLOAT3(0.0f, 0.0f, 0.0f);
	deviceContext->Unmap(sobelBuffer, 0);
	deviceContext->DSSetConstantBuffers(1, 1, &sobelBuffer);
	deviceContext->DSSetShaderResources(0, 1, &displacementTexture);
	deviceContext->DSSetSamplers(0, 1, &sampleState);

	//Map the dynamic tessellation buffer
	deviceContext->Map(dynamicTessBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource);
	DynamicTessType* dynamicPtr = (DynamicTessType*)mappedResource.pData;
	dynamicPtr->cameraPos = cameraPos;
	dynamicPtr->minTessFactor = minTessFactor;
	dynamicPtr->maxTessFactor = maxTessFactor;
	deviceContext->Unmap(dynamicTessBuffer, 0);
	deviceContext->HSSetConstantBuffers(0, 1, &dynamicTessBuffer);

	//Set the pixel shader textures and samplers
	deviceContext->PSSetShaderResources(0, 1, &textureMap);
	deviceContext->PSSetShaderResources(1, 1, &grassTexture);
	deviceContext->PSSetShaderResources(2, 1, &rockTexture);
	deviceContext->PSSetShaderResources(3, 1, &dirtTexture);
	deviceContext->PSSetShaderResources(4, 1, &sandTexture);
	deviceContext->PSSetShaderResources(5, 1, &sunShadowMap);
	deviceContext->PSSetShaderResources(6, 1, &spotShadowMap);
	deviceContext->PSSetShaderResources(7, 1, hillShadowMaps);
	deviceContext->PSSetSamplers(0, 1, &sampleState);
	deviceContext->PSSetSamplers(1, 1, &sampleStateShadow);
}


